import datetime
import json
import requests
from django.conf import settings as django_settings
from django.db.models import Sum, Count
from django.http import JsonResponse
from django.shortcuts import get_object_or_404
from rest_framework import viewsets
from rest_framework.decorators import action
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response

from customer.models import Customer
from invoice.invoice_serializer import InvoiceSerializer
from invoice.models import Invoice, PaymentMethod
from invoice.utils import create_items
from items.models import Item
from django.core.serializers.json import DjangoJSONEncoder

from kappi.settings import EMAIL_HOST_USER
from settings.models import Settings
from .utils import pdf_to_base64
from django.core import mail
from django.core.mail import send_mail
from django.template import loader
import logging

import os


class InvoiceViewSet(viewsets.ModelViewSet):
    @action(detail=True, methods=['post', 'put'], url_path='confirm')
    def confirm_invoice(self, request, pk=None):
            """
            Confirme une facture (proforma -> facture) et la synchronise à la DGI.
            """
            logger = logging.getLogger("invoice.dgi")

            try:
                invoice = self.get_object()
                # UID à confirmer: depuis la requête ou l'invoice
                uid = request.data.get('uid') or invoice.external_invoice_id
                if not uid:
                    return Response({"detail": "Missing DGI uid to confirm", "hint": "Provide 'uid' or ensure 'external_invoice_id' is set on the invoice."}, status=400)

                # Calcul des totaux: on accepte override via request, avec conversion robuste
                def _to_float(val, default):
                    try:
                        if val is None or val == "":
                            return float(default)
                        return float(val)
                    except Exception:
                        return float(default)

                subtotal = _to_float(getattr(invoice, 'subtotal', 0), 0)
                labor = _to_float(getattr(invoice, 'labor', 0), 0)
                tva_rate = _to_float(getattr(invoice, 'tva_rate', 0), 0)
                computed_vtotal = subtotal * tva_rate / 100.0
                vtotal = _to_float(request.data.get('vtotal'), computed_vtotal)
                total = _to_float(request.data.get('total'), (subtotal + labor + vtotal))
                confirm_payload = {"total": float(total), "vtotal": float(vtotal)}

                # Envoi à la DGI: PUT /api/invoice/{uid}/confirm
                dgi_token = getattr(django_settings, 'DGI_API_TOKEN', None)
                dgi_url_base = getattr(django_settings, 'DGI_API_URL', 'https://developper.dgirdc.cd/edef/api/invoice')
                confirm_url = f"{dgi_url_base}/{uid}/confirm"
                headers = {"Content-Type": "application/json-patch+json", "Accept": "text/plain"}
                if dgi_token:
                    headers["Authorization"] = f"Bearer {dgi_token}"
                else:
                    return Response({"detail": "No DGI token configured (DGI_API_TOKEN)", "sent_confirm_payload": confirm_payload}, status=400)

                dgi_response = requests.put(confirm_url, json=confirm_payload, headers=headers, timeout=15)
                logger.info("[DGI] Confirm PUT status: %s", getattr(dgi_response, 'status_code', None))
                try:
                    logger.info("[DGI] Confirm PUT body: %s", dgi_response.text)
                except Exception:
                    pass

                # Réponse
                if dgi_response.status_code == 200:
                    # Essayer de parser JSON sinon renvoyer texte
                    try:
                        body = dgi_response.json()
                    except Exception:
                        body = {"text": dgi_response.text}
                    # Mettre à jour l'état local seulement après succès DGI
                    invoice.proforma_state = False
                    invoice.save()
                    return Response({
                        "detail": "Invoice confirmed at DGI",
                        "uid": uid,
                        "dgi_status": 200,
                        "dgi_response": body,
                        "sent_confirm_payload": confirm_payload,
                        "proforma_state": invoice.proforma_state
                    })
                else:
                    return Response({
                        "detail": "DGI confirm error",
                        "uid": uid,
                        "dgi_status": dgi_response.status_code,
                        "dgi_response": dgi_response.text,
                        "sent_confirm_payload": confirm_payload,
                        "confirm_url": confirm_url
                    }, status=400)
            except Exception as e:
                logger.error("[DGI] Exception confirm_invoice: %s", str(e), exc_info=True)
                return Response({
                    "error": str(e),
                    "exception": e.__class__.__name__,
                    "detail": "Error confirming invoice"
                }, status=500)
    queryset = Invoice.objects.all()
    serializer_class = InvoiceSerializer
    permission_classes = [IsAuthenticated]

    def create(self, request):
        def _create_impl():
            import requests
            from django.utils import timezone
            import logging
            logger = logging.getLogger("invoice.dgi")

            # Valider les champs requis
            try:
                required_checks = [
                    (request.data.get('items'), "items required"),
                    (request.data.get('proforma_state') is not None, "proforma_state required"),
                    (request.data.get('customer'), "customer required"),
                    (request.data.get('method'), "method required"),
                    (request.data.get('create_invoice_at'), "create_invoice_at required"),
                    (request.data.get('invoice_expire_at'), "invoice_expire_at required"),
                    (request.data.get('model_invoice') is not None and request.data.get('model_invoice') >= 0, "model_invoice required"),
                    (request.data.get('labor') is not None and request.data.get('labor') >= 0, "labor required"),
                    (request.data.get('label_model') is not None, "label_model required"),
                    (request.data.get('payment_rules') is not None, "payment_rules required"),
                    (request.data.get('currency'), "currency required"),
                ]
                
                if not all(check[0] for check in required_checks):
                    missing = [check[1] for check in required_checks if not check[0]]
                    return JsonResponse({"error": f"Missing fields: {', '.join(missing)}"}, status=400)

                payment_method = get_object_or_404(PaymentMethod, id=request.data.get('method'))
                customer = Customer.objects.filter(name=request.data.get('customer'), user=request.user,
                                                   delete_state=False).exists()
                if not customer:
                    customer = Customer.objects.create(user=request.user, name=request.data.get('customer'))
                    customer.save()
                else:
                    customer = get_object_or_404(Customer, name=request.data.get('customer'), user=request.user,
                                                 delete_state=False)
                # Mettre à jour email/phone du client local à partir du payload
                client_info = request.data.get('client')
                if isinstance(client_info, dict):
                    email_val = client_info.get('contact')
                    phone_val = client_info.get('phone') or client_info.get('tel')
                    changed = False
                    if email_val and getattr(customer, 'email', None) != email_val:
                        customer.email = email_val
                        changed = True
                    if phone_val and getattr(customer, 'phone', None) != phone_val:
                        customer.phone = phone_val
                        changed = True
                    if changed:
                        customer.save()
                items_list = []
                total_sum = 0
                pourcent_tva = 0
                total_sum_tva = 0
                for item in request.data.get('items'):
                    items_list.append(json.dumps(item))
                    item_exist = Item.objects.filter(label=item['name']).exists()
                    total_sum += item['price'] * item['quantity']
                    if item_exist is True:
                        continue
                    else:
                        create_items(request.user, item['name'], item['price'], item.get('description', ''))

                if request.data.get('tva_rate'):
                    pourcent_tva += float(total_sum) * float(request.data.get('tva_rate')) / 100
                    total_sum_tva += total_sum + pourcent_tva + request.data.get('labor')
                else:
                    total_sum_tva += total_sum + request.data.get('labor')

                # Construction du payload pour l'API DGI (priorité aux valeurs du payload du client)
                # Préparer client/operator avec complétions douces pour champs manquants
                client_payload = {}
                if isinstance(request.data.get("client"), dict):
                    client_payload = dict(request.data.get("client"))
                client_payload.setdefault("name", request.data.get("customer"))
                client_payload.setdefault("type", "PP")
                # Utiliser email/phone du client local si non fournis
                if getattr(customer, "email", None):
                    client_payload.setdefault("contact", customer.email)
                if getattr(customer, "phone", None):
                    client_payload.setdefault("phone", customer.phone)

                operator_payload = {}
                if isinstance(request.data.get("operator"), dict):
                    operator_payload = dict(request.data.get("operator"))
                operator_payload.setdefault("id", request.data.get("isf"))

                # Nettoyer les items pour matcher l'exemple DGI (pas de description)
                items_clean = []
                for item in request.data.get("items") or []:
                    item_clean = {k: v for k, v in item.items() if k in ["code", "name", "type", "price", "quantity", "taxGroup"]}
                    items_clean.append(item_clean)

                # Forcer le NIF de test et curRate null si non fourni
                dgi_nif = request.data.get("nif") or "A2324624R"
                dgi_cur_rate = request.data.get("curRate")
                if dgi_cur_rate in (None, "", "null"):
                    dgi_cur_rate = None

                # Date au format +01:00 (exemple DGI)
                from datetime import datetime, timezone, timedelta
                now = datetime.now(timezone(timedelta(hours=1)))
                dgi_cur_date = now.strftime("%Y-%m-%dT%H:%M:%S+01:00")

                dgi_payload = {
                    "nif": dgi_nif,
                    "rn": request.data.get("rn"),
                    "mode": request.data.get("mode") or "ttc",
                    "isf": request.data.get("isf"),
                    "type": request.data.get("type"),
                    "items": items_clean,
                    "client": client_payload if client_payload else {},
                    "operator": operator_payload if operator_payload else {},
                    "totals": request.data.get("totals") or {"total": total_sum_tva},
                    "curCode": (request.data.get("curCode") or request.data.get("currency") or "USD"),
                    "curDate": dgi_cur_date,
                    "curRate": dgi_cur_rate
                }

                # Correction automatique: harmoniser le total envoyé avec le total calculé
                try:
                    provided_total = None
                    if isinstance(request.data.get('totals'), dict):
                        provided_total = request.data.get('totals').get('total')
                    if provided_total is None or abs(float(provided_total) - float(total_sum_tva)) > 0.005:
                        dgi_payload["totals"] = {"total": float(total_sum_tva)}
                except Exception:
                    dgi_payload["totals"] = {"total": float(total_sum_tva)}

                # LOG du payload DGI
                logger.info("[DGI] Payload envoyé: %s", json.dumps(dgi_payload, ensure_ascii=False))

                # Appel à l'API DGI
                dgi_response = None
                dgi_invoice_id = None
                dgi_auth_source = "none"
                try:
                    # Token DGI depuis les settings Django
                    from django.conf import settings as django_settings
                    dgi_token = django_settings.DGI_API_TOKEN
                    headers = {"Content-Type": "application/json"}
                    if dgi_token:
                        headers["Authorization"] = f"Bearer {dgi_token}"
                        dgi_auth_source = "settings"
                        dgi_response = requests.post(
                            django_settings.DGI_API_URL,
                            json=dgi_payload,
                            headers=headers,
                            timeout=15
                        )
                        # LOG de la réponse DGI (status + body)
                        logger.info("[DGI] Réponse status: %s", getattr(dgi_response, 'status_code', None))
                        try:
                            logger.info("[DGI] Réponse body: %s", dgi_response.text)
                        except Exception:
                            pass
                    else:
                        # Pas de jeton en Settings: ne pas appeler DGI, retourner une info claire
                        dgi_response = {"error": "Missing DGI token in Django settings (DGI_API_TOKEN)"}
                        dgi_auth_source = "missing"
                # Certaines API renvoient 201 en cas de création
                    if hasattr(dgi_response, 'status_code') and dgi_response.status_code in (200, 201):
                        try:
                            body = dgi_response.json()
                            dgi_invoice_id = (
                                body.get("uid")
                                or body.get("id")
                                or body.get("invoiceId")
                                or body.get("invoice_id")
                            )
                        except Exception:
                            dgi_invoice_id = None
                except Exception as e:
                    logger.error("[DGI] Exception lors de l'appel: %s", str(e))
                    dgi_response = str(e)

                invoice = Invoice.objects.create(
                    user=request.user,
                    customer=customer,
                    items=items_list,
                    invoice_total_amount=total_sum_tva,
                    method=payment_method,
                    subtotal=total_sum,
                    create_invoice_at=request.data.get('create_invoice_at'),
                    invoice_expire_at=request.data.get('invoice_expire_at'),
                    footer_note=request.data.get('footer_note'),
                    tva_rate=request.data.get('tva_rate'),
                    proforma_state=request.data.get('proforma_state'),
                    model_invoice=request.data.get('model_invoice'),
                    labor=request.data.get('labor'),
                    label_model=request.data.get('label_model'),
                    currency=request.data.get('currency'),
                    payment_rules=request.data.get('payment_rules'),
                    external_invoice_id=dgi_invoice_id
                )
                # Sécuriser la sérialisation de la réponse DGI
                try:
                    dgi_status_code = getattr(dgi_response, 'status_code', None)
                except Exception:
                    dgi_status_code = None

                # Toujours retourner un contenu sérialisable (dict/str/None),
                # même en cas de statut HTTP >= 400 où Response est "falsy".
                if dgi_response is not None and hasattr(dgi_response, 'json') and hasattr(dgi_response, 'text'):
                    try:
                        dgi_response_body = dgi_response.json()
                    except Exception:
                        dgi_response_body = dgi_response.text
                else:
                    # Peut être une chaîne (exception) ou None
                    dgi_response_body = dgi_response

                return JsonResponse({
                    "detail": "success",
                    "id": invoice.id,
                    "user": request.user.id,
                    "customer": request.data.get('customer'),
                    "external_invoice_id": dgi_invoice_id,
                    "dgi_status_code": dgi_status_code,
                    "dgi_response": dgi_response_body,
                    "dgi_auth": dgi_auth_source,
                    # Aide au diagnostic: totaux calculés localement vs envoyés
                    "computed_total": total_sum_tva,
                    "sent_totals": dgi_payload.get("totals"),
                    # Echo du payload envoyé (sans header Authorization)
                    "dgi_payload_sent": dgi_payload
                })
            except Exception as e:
                logger.error("[Invoice Create] Exception: %s", str(e), exc_info=True)
                return JsonResponse({
                    "error": str(e),
                    "detail": "Error creating invoice"
                }, status=500)

        return _create_impl()

    def update(self, request, *args, **kwargs):
        instance = self.get_object()
        # Validation minimale
        required_fields = ['items', 'customer', 'method', 'create_invoice_at', 'invoice_expire_at', 'tva_rate', 'model_invoice', 'labor', 'label_model', 'currency', 'payment_rules', 'proforma_state']
        missing = [f for f in required_fields if request.data.get(f) in (None, '')]
        if missing:
            return JsonResponse({"error": f"Missing fields: {', '.join(missing)}"}, status=400)

        items_list = []
        total_sum = 0
        for item in request.data.get('items'):
            items_list.append(json.dumps(item))
            total_sum += float(item['price']) * float(item['quantity'])
            if not Item.objects.filter(label=item['name']).exists():
                create_items(request.user, item['name'], item['price'], item.get('description', ''))

        # Calcul TVA
        pourcent_tva = 0
        total_sum_tva = 0
        if request.data.get('tva_rate'):
            pourcent_tva = float(total_sum) * float(request.data.get('tva_rate')) / 100
            total_sum_tva = total_sum + pourcent_tva + float(request.data.get('labor'))
        else:
            total_sum_tva = total_sum + float(request.data.get('labor'))

        payment_method = get_object_or_404(PaymentMethod, id=request.data.get('method'))
        customer = Customer.objects.filter(name=request.data.get('customer'), user=request.user, delete_state=False).first()
        if not customer:
            customer = Customer.objects.create(user=request.user, name=request.data.get('customer'))

        # Mise à jour
        instance.items = items_list
        instance.customer = customer
        instance.invoice_total_amount = total_sum_tva
        instance.method = payment_method
        instance.create_invoice_at = request.data.get('create_invoice_at')
        instance.invoice_expire_at = request.data.get('invoice_expire_at')
        instance.tva_rate = request.data.get('tva_rate')
        instance.subtotal = total_sum
        instance.proforma_state = request.data.get('proforma_state')
        instance.model_invoice = request.data.get('model_invoice')
        instance.label_model = request.data.get('label_model')
        instance.labor = request.data.get('labor')
        instance.currency = request.data.get('currency')
        instance.payment_rules = request.data.get('payment_rules')
        instance.footer_note = request.data.get('footer_note')
        instance.save()

        return JsonResponse({
            "detail": "success",
            "id": instance.id,
            "user": request.user.id,
            "customer": instance.customer.name,
            "external_invoice_id": instance.external_invoice_id,
            "footer_note": instance.footer_note,
            "paid": instance.paid,
            "send": instance.send,
            "method": instance.method.id,
            "proforma_state": instance.proforma_state,
            "model_invoice": instance.model_invoice,
            "currency": instance.currency,
            "payment_rules": instance.payment_rules,
            "labor": instance.labor,
            "label_model": instance.label_model
        })

    def get_queryset(self):
        """
        This view should return a list of all the purchases
        for the currently authenticated user.
        """
        user = self.request.user
        qs = Invoice.objects.filter(user=user).order_by("-id")
        # Filtre optionnel par identifiant externe (UID DGI)
        ext_id = self.request.query_params.get('external_invoice_id')
        if ext_id:
            qs = qs.filter(external_invoice_id=ext_id)
        return qs

    @action(methods=["GET"], detail=False)
    def invoice_paid(self, request):
        invoices = Invoice.objects.filter(proforma_state=False, user=request.user, paid=True,
                                          year_of_invoice=datetime.date.today().year).values(
            'month_of_invoice', 'year_of_invoice', 'send').annotate(number_of_invoice=Count('id'))
        list_stat_invoice = []
        for invoice in invoices:
            list_stat_invoice.append({
                "month_of_invoice": invoice["month_of_invoice"],
                "year_of_invoice": invoice["year_of_invoice"],
                "number_of_invoice": invoice["number_of_invoice"],
                "state_send": invoice["send"],
            })

        if len(list_stat_invoice) > 0 < 12:
            for i in range(list_stat_invoice[len(list_stat_invoice) - 1]['month_of_invoice'], 12):
                list_stat_invoice.append({
                    "month_of_invoice": i + 1,
                    "year_of_invoice": datetime.date.today().year,
                    "number_of_invoice": 0,
                    "state_send": True,
                })

            if len(list_stat_invoice) < 12:
                for i in range(list_stat_invoice[0]['month_of_invoice'], 1, -1):
                    list_stat_invoice.append({
                        "month_of_invoice": i - 1,
                        "year_of_invoice": datetime.date.today().year,
                        "number_of_invoice": 0,
                        "state_send": True,
                    })

        return JsonResponse(list_stat_invoice, safe=False)

    @action(methods=["GET"], detail=False)
    def invoice_not_paid(self, request):
        invoices = Invoice.objects.filter(proforma_state=False, user=request.user, paid=False,
                                          year_of_invoice=datetime.date.today().year).values(
            'month_of_invoice', 'year_of_invoice', 'send').annotate(number_of_invoice=Count('id'))
        list_stat_invoice = []
        for invoice in invoices:
            list_stat_invoice.append({
                "month_of_invoice": invoice["month_of_invoice"],
                "year_of_invoice": invoice["year_of_invoice"],
                "number_of_invoice": invoice["number_of_invoice"],
                "state_send": invoice["send"],
            })

        if len(list_stat_invoice) > 0 < 12:
            for i in range(list_stat_invoice[len(list_stat_invoice) - 1]['month_of_invoice'], 12):
                list_stat_invoice.append({
                    "month_of_invoice": i + 1,
                    "year_of_invoice": datetime.date.today().year,
                    "number_of_invoice": 0,
                    "state_send": True,
                })

            if len(list_stat_invoice) < 12:
                for i in range(list_stat_invoice[0]['month_of_invoice'], 1, -1):
                    list_stat_invoice.append({
                        "month_of_invoice": i - 1,
                        "year_of_invoice": datetime.date.today().year,
                        "number_of_invoice": 0,
                        "state_send": True,
                    })
        return JsonResponse(list_stat_invoice, safe=False)

    @action(methods=["GET"], detail=False)
    def invoice_send(self, request):
        invoices = Invoice.objects.filter(proforma_state=False, user=request.user, send=True,
                                          year_of_invoice=datetime.date.today().year).values('month_of_invoice',
                                                                                             'year_of_invoice',
                                                                                             'send').annotate(
            number_of_invoice=Count('id'))
        list_stat_invoice = []
        for invoice in invoices:
            list_stat_invoice.append({
                "month_of_invoice": invoice["month_of_invoice"],
                "year_of_invoice": invoice["year_of_invoice"],
                "number_of_invoice": invoice["number_of_invoice"],
                "state_send": invoice["send"],
            })

        if len(list_stat_invoice) > 0 < 12:
            for i in range(list_stat_invoice[len(list_stat_invoice) - 1]['month_of_invoice'], 12):
                list_stat_invoice.append({
                    "month_of_invoice": i + 1,
                    "year_of_invoice": datetime.date.today().year,
                    "number_of_invoice": 0,
                    "state_send": True,
                })

            if len(list_stat_invoice) < 12:
                for i in range(list_stat_invoice[0]['month_of_invoice'], 1, -1):
                    list_stat_invoice.append({
                        "month_of_invoice": i - 1,
                        "year_of_invoice": datetime.date.today().year,
                        "number_of_invoice": 0,
                        "state_send": True,
                    })

            if len(list_stat_invoice) < 12:
                for i in range(list_stat_invoice[0]['month_of_invoice'], 1, -1):
                    list_stat_invoice.append({
                        "month_of_invoice": i - 1,
                        "year_of_invoice": datetime.date.today().year,
                        "number_of_invoice": 0,
                        "state_send": True,
                    })

                if len(list_stat_invoice) < 12:
                    for i in range(list_stat_invoice[0]['month_of_invoice'], 1, -1):
                        list_stat_invoice.append({
                            "month_of_invoice": i - 1,
                            "year_of_invoice": datetime.date.today().year,
                            "number_of_invoice": 0,
                            "state_send": True,
                        })

        return JsonResponse(list_stat_invoice, safe=False)

    @action(methods=["GET"], detail=False)
    def invoice_not_send(self, request):
        invoices = Invoice.objects.filter(proforma_state=False, user=request.user, send=False,
                                          year_of_invoice=datetime.date.today().year).values(
            'month_of_invoice', 'year_of_invoice', 'send').annotate(number_of_invoice=Count('id'))
        list_stat_invoice = []
        for invoice in invoices:
            list_stat_invoice.append({
                "month_of_invoice": invoice["month_of_invoice"],
                "year_of_invoice": invoice["year_of_invoice"],
                "number_of_invoice": invoice["number_of_invoice"],
                "state_send": invoice["send"],
            })

        if len(list_stat_invoice) > 0 < 12:
            for i in range(list_stat_invoice[len(list_stat_invoice) - 1]['month_of_invoice'], 12):
                list_stat_invoice.append({
                    "month_of_invoice": i + 1,
                    "year_of_invoice": datetime.date.today().year,
                    "number_of_invoice": 0,
                    "state_send": False,
                })

            if len(list_stat_invoice) < 12:
                for i in range(list_stat_invoice[0]['month_of_invoice'], 1, -1):
                    list_stat_invoice.append({
                        "month_of_invoice": i - 1,
                        "year_of_invoice": datetime.date.today().year,
                        "number_of_invoice": 0,
                        "state_send": False,
                    })

        return JsonResponse(list_stat_invoice, safe=False)

    @action(methods=['GET'], detail=False)
    def customers_invoice(self, request):
        customers = Invoice.objects.filter(proforma_state=False, user=request.user,currency="USD").values('customer__name').annotate(
            amount=Sum('invoice_total_amount'))[:5]
        list_stat_customers = []
        for customer in customers:
            list_stat_customers.append({
                "customer": customer["customer__name"],
                "amount": int(customer["amount"])
            })

        return JsonResponse(list_stat_customers, safe=False)

    @action(methods=['GET'], detail=False)
    def customers_invoice_cdf(self, request):
        customers = Invoice.objects.filter(proforma_state=False, user=request.user, currency="CDF").values(
            'customer__name').annotate(
            amount=Sum('invoice_total_amount'))[:5]
        list_stat_customers = []
        for customer in customers:
            list_stat_customers.append({
                "customer": customer["customer__name"],
                "amount": int(customer["amount"])
            })

        return JsonResponse(list_stat_customers, safe=False)

    @action(methods=['POST'], detail=False)
    def change_paid_state_invoice(self, request, *args, **kwargs):

        if request.data.get('id') and request.data.get('paid') == True or request.data.get('paid') == False:
            invoice = Invoice.objects.get(user=request.user, id=request.data.get('id'))
            invoice.paid = request.data.get('paid')
            invoice.save()
            serializer = InvoiceSerializer([invoice], many=True)
            return JsonResponse(serializer.data[0], safe=False)
        else:
            return Response({
                "id": "id for invoice is required",
                "paid": "is required true or false  value"
            })

    @action(methods=['GET'], detail=False)
    def change_send_state_invoice(self, request, *args, **kwargs):

        if request.data.get('id') and request.data.get('send') == True or request.data.get('send') == False:
            invoice = Invoice.objects.get(user=request.user, id=request.data.get('id'))
            invoice.send = request.data.get('send')
            invoice.save()
            serializer = InvoiceSerializer([invoice], many=True)
            return JsonResponse(serializer.data[0], safe=False)
        else:
            return Response({
                "id": "id for invoice is required",
                "send": "is required true or false  value"
            })

    @action(methods=['GET'], detail=False)
    def invoice_per_month(self, request):

        invoice_list = []
        for filter_month in range(1, 13):
            invoice = (Invoice.objects.filter(proforma_state=False, user=request.user, month_of_invoice=filter_month,
                                              year_of_invoice=datetime.date.today().year).values(
                'month_of_invoice', 'invoice_total_amount', 'year_of_invoice').annotate(
                number_of_invoice=Count('id'))).exists()

            invoice_data = (Invoice.objects.filter(user=request.user, month_of_invoice=filter_month,
                                                   year_of_invoice=datetime.date.today().year).values(
                'month_of_invoice', 'year_of_invoice').annotate(number_of_invoice=Count('id'),
                                                                total=Sum('invoice_total_amount')))

            if invoice:
                for invoice in invoice_data:
                    invoice_list.append({
                        "month_of_invoice": invoice["month_of_invoice"],
                        "year_of_invoice": invoice["year_of_invoice"],
                        "number_of_invoice": invoice["number_of_invoice"],
                        "invoice_total_amount": int(invoice["total"]),
                    })
            else:
                invoice_list.append({
                    "month_of_invoice": filter_month,
                    "year_of_invoice": datetime.date.today().year,
                    "number_of_invoice": 0,
                    "invoice_total_amount": 0,
                })
        return Response(invoice_list)

    @action(methods=['GET','POST'], detail=False)
    def change_proforma_state_invoice(self, request, *args, **kwargs):

        if request.data.get('id') and request.data.get('proforma_state') == True or request.data.get(
                'proforma_state') == False:
            invoice = Invoice.objects.get(user=request.user, id=request.data.get('id'))
            invoice.proforma_state = request.data.get('proforma_state')
            invoice.save()
            serializer = InvoiceSerializer([invoice], many=True)
            return JsonResponse(serializer.data[0], safe=False)
        else:
            return Response({
                "id": "id for invoice is required",
                "proforma_state": "is required true or false  value"
            })

    @action(methods=['GET'], detail=False)
    def proforma_invoice_per_month(self, request):

        invoice_list = []
        for filter_month in range(1, 13):
            invoice = (Invoice.objects.filter(proforma_state=True, user=request.user, month_of_invoice=filter_month,
                                              year_of_invoice=datetime.date.today().year).values(
                'month_of_invoice', 'invoice_total_amount', 'year_of_invoice').annotate(
                number_of_invoice=Count('id'))).exists()

            invoice_data = (
                Invoice.objects.filter(proforma_state=True, user=request.user, month_of_invoice=filter_month,
                                       year_of_invoice=datetime.date.today().year).values(
                    'month_of_invoice', 'year_of_invoice').annotate(number_of_invoice=Count('id'),
                                                                    total=Sum('invoice_total_amount')))

            if invoice:
                for invoice in invoice_data:
                    invoice_list.append({
                        "month_of_invoice": invoice["month_of_invoice"],
                        "year_of_invoice": invoice["year_of_invoice"],
                        "number_of_invoice": invoice["number_of_invoice"],
                        "invoice_total_amount": int(invoice["total"]),
                    })
            else:
                invoice_list.append({
                    "month_of_invoice": filter_month,
                    "year_of_invoice": datetime.date.today().year,
                    "number_of_invoice": 0,
                    "invoice_total_amount": 0,
                })

        return Response(invoice_list)

    @action(methods=['POST'], detail=False)
    def invoice_import(self, request):
        if request.data.get('invoice'):

            count_row = 0
            method = PaymentMethod.objects.get(id=1)

            for index in range(len(request.data.get('invoice'))):

                try:

                    if request.data.get('invoice')[index]['customer'] and request.data.get('invoice')[index][
                        'items'] and request.data.get('invoice')[index]['paid'] >= 0 and \
                            request.data.get('invoice')[index]['send'] >= 0 and request.data.get('invoice')[index][
                        'create_invoice_at'] and request.data.get('invoice')[index]['invoice_expire_at'] and \
                            request.data.get('invoice')[index]['month_of_invoice'] and \
                            request.data.get('invoice')[index][
                                'invoice_total_amount'] and request.data.get('invoice')[index]['tva_rate'] >= 0 and \
                            request.data.get('invoice')[index]['subtotal'] and request.data.get('invoice')[index][
                        'proforma_state'] >= 0:
                        items = []
                        count_row += 1
                        for item in request.data.get('invoice')[index]['items']:
                            items.append(json.dumps({
                                "name": item['name'],
                                "price": item['price'],
                                "quantity": item['quantity'],
                                "description": "Description"
                            }))
                        if Customer.objects.filter(name=request.data.get('invoice')[index]["customer"]).exists():
                            customer = Customer.objects.get(name=request.data.get('invoice')[index]["customer"])
                        else:
                            customer = Customer.objects.create(user=request.user,
                                                               name=request.data.get('invoice')[index]["customer"])
                            customer.save()
                        invoice_create = Invoice.objects.create(
                            user=request.user,
                            customer=customer,
                            paid=request.data.get('invoice')[index]["paid"],
                            send=request.data.get('invoice')[index]["send"],
                            invoice_total_amount=request.data.get('invoice')[index]["invoice_total_amount"],
                            method=method,
                            create_invoice_at=request.data.get('invoice')[index]["create_invoice_at"],
                            invoice_expire_at=request.data.get('invoice')[index]["invoice_expire_at"],
                            month_of_invoice=request.data.get('invoice')[index]["month_of_invoice"],
                            tva_rate=request.data.get('invoice')[index]["tva_rate"],
                            subtotal=request.data.get('invoice')[index]["subtotal"],
                            proforma_state=request.data.get('invoice')[index]["proforma_state"],
                            items=items
                        )
                        invoice_create.save()

                except Exception as e:
                    return Response({
                        "customer": "is required",
                        "items": "is required",
                        "paid": "is required",
                        "send": "is required",
                        "create_invoice_at": "is required",
                        "invoice_expire_at": "is required",
                        "month_of_invoice": "is required",
                        "tva_rate": "is required",
                        "subtotal": "is required",
                        "proforma_state": "is required",
                        "method": "is required"
                    })
            return Response({
                "status": "success",
                "row_register": count_row
            })

        else:
            return Response({
                "status": "error",
                "invoice": "is required | List invoice:[{}]"
            })

    @action(methods=['GET'], detail=False)
    def get_invoice_per_customer(self, request):

        if request.data.get('customer'):
            print(request.data.get('customer'))
            customer = Customer.objects.get(id=request.data.get('customer'), user=request.user)
            invoice = list(Invoice.objects.filter(customer=customer, user=request.user).values())
            return JsonResponse({"invoice": invoice}, safe=False)
        else:
            return JsonResponse({"customer": "is required"}, safe=False)

    @action(detail=False, methods=['POST'])
    def register_list_of_invoices(self, request):
        if request.data.get('invoice'):
            payment_method = get_object_or_404(PaymentMethod, id=1)
            items_list = []
            total_sum = 0
            pourcent_tva = 0
            total_sum_tva = 0
            for invoice in request.data.get('invoice'):
                customer = Customer.objects.filter(name=invoice['customer'], user=request.user,
                                                   delete_state=False).exists()
                if not customer:
                    customer = Customer.objects.create(user=request.user, name=invoice['customer'])
                    customer.save()
                else:
                    customer = get_object_or_404(Customer, name=invoice['customer'], user=request.user,
                                                 delete_state=False)
                for item in invoice['items']:
                    print(item)
                    items_list.append(json.dumps(item))
                    item_exist = Item.objects.filter(label=item['name']).exists()
                    total_sum += item['price'] * item['quantity']
                    if item_exist is True:
                        continue
                    else:
                        create_items(request.user, item['name'], item['price'], item['description'])

                if invoice['tva_rate']:
                    pourcent_tva += float(total_sum) * float(invoice['tva_rate']) / 100
                    total_sum_tva += total_sum + pourcent_tva + float(invoice['labor'])
                else:
                    total_sum_tva += total_sum + invoice['labor']

                invoice = Invoice.objects.create(
                        user=request.user,
                        customer=customer,
                        items=items_list,
                        invoice_total_amount=total_sum_tva,
                        method=payment_method,
                        subtotal=total_sum,
                        create_invoice_at=invoice['create_invoice_at'],
                        invoice_expire_at=invoice['invoice_expire_at'],
                        footer_note=invoice['footer_note'],
                        tva_rate=invoice['tva_rate'],
                        proforma_state=invoice['proforma_state'],
                        model_invoice=invoice['model_invoice'],
                        labor=invoice['labor'],
                        label_model=invoice['label_model'],
                        currency=invoice['currency'],
                        payment_rules=invoice['payment_rules'],

                    )
                invoice.save()

            return JsonResponse({
                "detail": "success",
                "user": request.user.id,
            })
        else:
            return JsonResponse({"invoice": "list of invoices [{}]"})

    @action(methods=['POST'], detail=False)
    def share_invoice_by_mail_to_customer(self, request):
        email = request.data.get('email')
        subject = request.data.get('subject')
        message = request.data.get('message')
        attach_file64_string = request.data.get('attach_file')

        if not email or not message or not attach_file64_string or not subject:
            return Response({
                "email": "is required",
                "message": "is required",
                "subject": "is required",
                "attach_file": "is required",
            }, status=400)

        try:
            connection = mail.get_connection()
            connection.open()

            invoice_path = pdf_to_base64(attach_file64_string)
            link_message = invoice_path['final']

            messages = link_message
            setting = get_object_or_404(Settings,user=request.user)
            html_message = loader.render_to_string(
                'email/email_of_customers.html', locals())
            send_mail(
                subject=subject,
                message=messages,
                from_email=EMAIL_HOST_USER,
                recipient_list=[email],
                fail_silently=True,
                html_message=html_message
            )
            return Response({
                "detail": "email sent",
                "success": True
            }, status=200)

        except Exception as e:
            return Response({"error": str(e)}, status=500)

